<?php

namespace HS\Providers;

use HS\MultiPortal;

use Illuminate\Support\Env;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\ServiceProvider;
use Illuminate\Database\QueryException;
use Illuminate\Queue\Events\JobProcessed;

use Dotenv\Dotenv;
use Dotenv\Exception\InvalidPathException;
use Bugsnag\BugsnagLaravel\Facades\Bugsnag;

class AppServiceProvider extends ServiceProvider
{
    /**
     * Bootstrap any application services.
     *
     * @return void
     */
    public function boot()
    {
        if (config('helpspot.profile') && ! app()->runningInConsole()) {
            \DB::connection()->enableQueryLog();
        }

        // These variables are only set on error, so it's likely
        // after we finish bootstrapping
        Bugsnag::registerCallback(function ($report) {
            $report->setMetaData([
                'account' => [
                    'name' => hs_setting('cHD_ORGNAME', 'unknown'),
                    'org_id' => hs_setting('cHD_CUSTOMER_ID', 1),
                    'version' => hs_setting('cHD_VERSION', 'v5.unknown'),
                ],
            ]);
        });

        if (! defined('cBASEPATH')) {
            define('cBASEPATH', base_path('helpspot'));
        }

        if (! defined('cHOST')) {
            define('cHOST', rtrim(config('app.url'), '/'));
        }

        // If we want to force maintenance mode via "MAINTENANCE_MODE=true" in .env
        // then we set "artisan down" here. The command can be safely be run run
        // if already in maintenance mode
        if (config('helpspot.maintenance_mode', false)) {
            Artisan::call('down');
        }

        if (config('helpspot.force_https')) {
            URL::forceScheme('https');
        }

        // If we're using the database cache driver, but the table
        // doesn't exist yet, fallback to file driver
        app()->resolving('cache', function() {
            if (config('cache.default') == 'database') {
                if (! Schema::hasTable(config('cache.stores.database.table'))) {
                    config()->set('cache.default', 'file');
                }
            }
        });

        // If we have a portal being used as a primary portal
        if (! defined('cMULTIPORTAL')) {
            try {
                $primaryPortal = MultiPortal::asPrimary()->first();
                if ($primaryPortal) {
                    define('cMULTIPORTAL', $primaryPortal->xPortal);
                }
            } catch(\Exception $e) {
                // swallow so we don't fail on upgrades.
            }
        }

        // Secondary portals (not used as primary) can optionally
        // have their own .env file that over-rides the base .env file
        if (defined('cMULTIPORTAL')) {
            $portal = MultiPortal::active()
                ->where('xPortal', cMULTIPORTAL)
                ->where('fIsPrimaryPortal', 0)
                ->first();

            if ($portal && is_dir($portal->sPortalPath) && $portal->sPortalPath != public_path()) {
                try {
                    $env = Dotenv::create(Env::getRepository(), $portal->sPortalPath);
                    $env->load();
                } catch (InvalidPathException $e) {
                    // swallow if portal directory does not exist
                }
            }
        }

        // Util has initDB and initSettings
        // initSettings calls items in api.lib.php.
        // ADODB calls some utf8 helpers
        //   The dependency chain stops there. Phew.
        require_once cBASEPATH.'/helpspot/lib/utf8.lib.php';
        require_once cBASEPATH.'/helpspot/lib/api.lib.php';
        require_once cBASEPATH.'/helpspot/lib/util.lib.php';

        // Database must be loaded before other init methods
        $this->initHelpspotDatabase();

        try {
            // Successful loading of HS_Settings determines
            // if HelpSpot is installed or not
            $this->initHelpspotSettings();

            Bugsnag::setAppVersion(hs_setting('cHD_VERSION'));

            $this->setSMTPConfig(); // Relies on HS_Settings loading
            if (! defined('HELPSPOT_INSTALLED')) {
                define('HELPSPOT_INSTALLED', true);
            }
        } catch (QueryException $e) {
            // HTTP Middleware will handle redirect to web-based installer
            if (! defined('HELPSPOT_INSTALLED')) {
                define('HELPSPOT_INSTALLED', false);
            }
        }

        //ensure compatibility with MySQL 5.6
        Schema::defaultStringLength(191);

        $this->initHelpSpotOrg(); // Has a fallback in case setting is not loaded from HS_Settings
        $this->initLanguage();

        Event::listen(JobProcessed::class, function(JobProcessed $event) {
            $jobHandler = $event->job->payload()['displayName'] ?? '';

            // If we've just sent a message, and the queue is "high", then sleep certain amount of seconds
            if (strpos($jobHandler, 'SendMessage') !== false && $event->job->getQueue() == 'high') {
                $secondsBetweenJobs = config('helpspot.send_mail_delay');
                if (is_numeric($secondsBetweenJobs) && $secondsBetweenJobs > 0) {
                    sleep((int)$secondsBetweenJobs);
                }
            }
        });
    }

    /**
     * Register any application services.
     *
     * @return void
     */
    public function register()
    {
        /*
         * HelpSpot Database Bootstrapping for our updated
         * Query Grammars, which help handle field types
         * not otherwise handled by laravel's database
         */
        $this->app->resolving('db', function ($db) {
            try {
                $grammar = defaultConnectionDetail('grammar');
                $db->connection()->setSchemaGrammar(new $grammar);
            } catch (\PDOException $e) {
                throw new \Exception('HelpSpot could not connect to the database, likely due to incorrect credentials supplied in .env. The error received from the database was: ' . $e->getMessage());
            }
        });
    }

    private function initHelpspotDatabase()
    {
        hsInitDB();
    }

    private function initHelpspotSettings()
    {
        hsInitSettings();
    }

    private function initHelpSpotOrg()
    {
        config()->set('app.name', hs_setting('cHD_ORGNAME', 'HelpSpot'));
    }

    private function setSMTPConfig()
    {
        if (hs_setting('cHD_MAIL_OUTTYPE') == 'mail') {
            config()->set('mail.default', 'sendmail');
        } else {
            // cHD_MAIL_OUTTYPE == 'smtp'
            $smtp = hs_unserialize(cHD_MAIL_SMTPCONN);

            // Get the app domain
            $defaultHelo = false;
            $domain = parse_url(hs_setting('cHOST'));
            if (isset($domain['host'])) {
                $defaultHelo = $domain['host'];
            }

            // Set these to null if we aren't authenticating
            $username = ($smtp['cHD_MAIL_SMTPAUTH']) ? $smtp['cHD_MAIL_SMTPUSER'] : null;
            $password = ($smtp['cHD_MAIL_SMTPAUTH']) ? $smtp['cHD_MAIL_SMTPPASS'] : null;

            config()->set('mail.default', 'smtp');
            config()->set('mail.mailers.smtp.host', $smtp['cHD_MAIL_SMTPHOST']);
            config()->set('mail.mailers.smtp.port', $smtp['cHD_MAIL_SMTPPORT']);
            config()->set('mail.mailers.smtp.encryption', $smtp['cHD_MAIL_SMTPPROTOCOL']);
            config()->set('mail.mailers.smtp.auth', $smtp['cHD_MAIL_SMTPAUTH']);
            config()->set('mail.mailers.smtp.username', $username);
            config()->set('mail.mailers.smtp.password', $password);
            config()->set('mail.mailers.smtp.timeout', ($smtp['cHD_MAIL_SMTPTIMEOUT']) ? $smtp['cHD_MAIL_SMTPTIMEOUT'] : null);
            config()->set('mail.mailers.smtp.local_domain', ($smtp['cHD_MAIL_SMTPHELO']) ? $smtp['cHD_MAIL_SMTPHELO'] : $defaultHelo);
            config()->set('mail.from', [
                'address' => cHD_NOTIFICATIONEMAILACCT,
                'name' => cHD_NOTIFICATIONEMAILNAME,
            ]);
        }
    }

    private function initLanguage()
    {
        $language = hs_setting('cHD_LANG', 'english-us');

        if (! is_dir(cBASEPATH.'/helpspot/lang/'.$language)) {
            $language = 'english-us';
        }

        require_once cBASEPATH.'/helpspot/lang/'.$language.'/lg.general.php';
        require_once cBASEPATH.'/helpspot/lang/'.$language.'/lg.pg.request.php';
    }
}
