<?php

use HS\Domain\Workspace\History;
use HS\Domain\Workspace\Document;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Database\Migrations\Migration;

class AddHistoryIdToDocuments extends Migration
{
    /**
     * Run the migrations.
     *
     * @return void
     */
    public function up()
    {

        // Don't make schema changes in large update mode
        // as we'll do this manually via pt-online-schema-change
        /*
         pt-online-schema-change \
            --alter "ADD COLUMN xRequestHistory INT(11) UNSIGNED DEFAULT NULL, ADD COLUMN xResponse INT(11) UNSIGNED DEFAULT NULL, ADD INDEX hs_documents_xrequesthistory_index (xRequestHistory)" \
            --ask-pass \
            h=hostname,u=user,A=utf8mb4,D=helpspot_db,t=HS_Documents # --execute
         */
        if (! config('helpspot.large_update_mode', false)) {
            Schema::table('HS_Documents', function (Blueprint $table) {
                if (! Schema::hasColumn('HS_Documents', 'xRequestHistory')) {
                    $table->unsignedInteger('xRequestHistory')
                        ->nullable()
                        ->index();
                }

                if (! Schema::hasColumn('HS_Documents', 'xResponse')) {
                    $table->unsignedInteger('xResponse')->nullable();
                }
            });
        }

        $this->migrateHistoryDocuments();
        $this->migrateResponses();
        $this->dropDocumentIdColumnFromHistory();
    }

    /**
     * Reverse the migrations.
     *
     * @return void
     */
    public function down()
    {
        Schema::table('HS_Documents', function (Blueprint $table) {
            $table->dropColumn('xRequestHistory');
            $table->dropColumn('xResponse');
        });
    }

    /**
     * Loop through all the request history and
     * update the Document with the parent history id.
     */
    protected function migrateHistoryDocuments()
    {
        $documentHistoryRows = collect([]);
        $affectedRequests = [];

        // Reassign document request history rows to their related document
        $chunk = DB::table('HS_Request_History')
            ->select(['dtGMTChange', 'xRequest', 'xDocumentId', 'xRequestHistory'])
            ->where('xDocumentId', '>', 0)
            ->orderBy('xRequestHistory', 'asc')
            ->limit(2000)
            ->get();

        $lastHistoryId = 0;

        while($chunk->count()) {
            foreach($chunk as $historyDocumentRow) {
                // Track history rows we will delete
                $documentHistoryRows->push($historyDocumentRow->xRequestHistory);

                // Count history rows we will delete per request
                if (isset($affectedRequests[$historyDocumentRow->xRequest])) {
                    $affectedRequests[$historyDocumentRow->xRequest] ++;
                } else {
                    $affectedRequests[$historyDocumentRow->xRequest] = 1;
                }

                $parent = History::where('xDocumentId', '=', 0)
                    ->where('dtGMTChange', '=', $historyDocumentRow->dtGMTChange)
                    ->where('xRequest', $historyDocumentRow->xRequest)
                    ->whereRaw("((tLog = '' OR tLog LIKE '%ccstaff%') OR tEmailHeaders <> '')")
                    ->orderBy('dtGMTChange', 'asc')
                    ->first();

                if ($parent) {
                    Document::where('xDocumentId', $historyDocumentRow->xDocumentId)
                        ->update(['xRequestHistory' => $parent->getKey()]);
                }

                $lastHistoryId = $historyDocumentRow->xRequestHistory;
            }

            $chunk = DB::table('HS_Request_History')
                ->select(['dtGMTChange', 'xRequest', 'xDocumentId', 'xRequestHistory'])
                ->where('xDocumentId', '>', 0)
                ->where('xRequestHistory', '>', $lastHistoryId)
                ->orderBy('xRequestHistory', 'asc')
                ->limit(2000)
                ->get();
        }


        // Large databases will get these deleted using the pt-archiver tool
        /*
         pt-archiver --source h=hostname,u=user,A=utf8mb4,D=helpspot_db,t=HS_Request_History \
             --ask-pass \
             --purge \
             --commit-each \
             --primary-key-only \
             --where "xDocumentId > 0" # --dry-run
         */
        if (! config('helpspot.large_update_mode', false)) {
            foreach ($documentHistoryRows->chunk(2000) as $chunk) {
                DB::table('HS_Request_History')
                    ->whereIn('xRequestHistory', $chunk->toArray())
                    ->delete();

                usleep(250000); // .25 seconds to allow the binlog to catch up
            }
        }

        // Fix read/unread on open, assigned requests.
        // Read/unread is based on the count of history rows and if the
        // assigned user has seen the request when it had X many history rows.
        // We deleted rows, so the count is now incorrect, resulting in
        // assigned staff seeing all open requests as unread.
        foreach($affectedRequests as $xRequest => $deletedHistoryCount) {
            DB::update(DB::raw("UPDATE HS_Request SET iLastReadCount = (iLastReadCount - $deletedHistoryCount)
    WHERE iLastReadCount - $deletedHistoryCount > 0
    AND fOpen = 1
    AND xPersonAssignedTo > 0
    AND xRequest = $xRequest"));
        }
    }

    /**
     * Loop through all the responses and
     * create the relationship with documents
     */
    protected function migrateResponses()
    {
        $responses = DB::table('HS_Responses')->get();
        foreach ($responses as $response) {
            $options = json_decode($response->tResponseOptions, true);
            if (isset($options['attachment']) && is_array($options['attachment'])) {
                foreach ($options['attachment'] as $attachId) {
                    $doc = Document::find($attachId);

                    if ($doc) {
                        $doc->xResponse = $response->xResponse;
                        $doc->save();
                    }
                }
            }
        }
    }

    /**
     * Drop the unused HS_Request_History.xDocumentId column
     */
    protected function dropDocumentIdColumnFromHistory()
    {
        // Don't make schema changes in large update mode
        // as we'll do this manually via pt-online-schema-change
        /*
             pt-online-schema-change --alter "DROP COLUMN xDocumentId" \
             --ask-pass \
             h=hostname,u=user,A=utf8mb4,D=helpspot_db,t=HS_Request_History # --execute
         */
        if (! config('helpspot.large_update_mode', false)) {
            if (Schema::hasColumn('HS_Request_History', 'xDocumentId')) {
                if (config('database.default') == 'sqlsrv') {
                    \HS\Database\SqlSrv::drop('HS_Request_History', 'xDocumentId');
                } else {
                    Schema::table('HS_Request_History', function (Blueprint $table) {
                        $table->dropColumn('xDocumentId');
                    });
                }
            }
        }
    }
}
