<?php

namespace HS\Auth\OAuth;

use Carbon\Carbon;
use Illuminate\Contracts\Support\Arrayable;

abstract class Credentials implements Arrayable
{
    /**
     * @var string
     */
    protected $token;

    /**
     * @var string
     */
    protected $refreshToken;

    /**
     * @var Carbon
     */
    protected $expiresAt;


    /**
     * Microsoft constructor.
     * @param string $token
     * @param string $refreshToken
     * @param Carbon $expiresAt
     */
    public function __construct($token, $refreshToken, Carbon $expiresAt)
    {
        $this->token = $token;
        $this->refreshToken = $refreshToken;
        $this->expiresAt = $expiresAt;
    }

    /**
     * Return a new Credentials instance from an encoded string
     * @param $encodedString
     * @return static
     */
    public static function fromEncoded($encodedString)
    {
        $credentials = json_decode(base64_decode($encodedString), JSON_THROW_ON_ERROR);

        return new static($credentials['token'], $credentials['refreshToken'], new Carbon($credentials['expiresAt']));
    }

    /**
     * Return an encoded string
     * @return string
     */
    public function toEncoded()
    {
        return base64_encode(json_encode($this->toArray(), true));
    }

    /**
     * @return string
     */
    public function token()
    {
        return $this->token;
    }

    /**
     * @return string
     */
    public function refreshToken()
    {
        return $this->refreshToken;
    }

    /**
     * @return Carbon
     */
    public function expiresAt()
    {
        return $this->expiresAt;
    }

    /**
     * @return bool
     */
    public function isExpired()
    {
        return $this->expiresAt->isPast();
    }

    /**
     * @return array
     */
    public function toArray()
    {
        return [
            'token' => $this->token,
            'refreshToken' => $this->refreshToken,
            'expiresAt' => $this->expiresAt->toISOString(),
        ];
    }

    /**
     * @return static
     */
    abstract public function refresh();
}
