<?php

namespace HS\Auth\OAuth;

use Carbon\Carbon;
use Firebase\JWT\JWT;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Str;

class Microsoft extends Credentials
{
    protected $scopes = [
        'offline_access',
        'openid',
        'profile',
        'user.read',

        'mail.send',
        'mail.read',
        'mail.readbasic',
        'mail.readwrite',
        'mailboxsettings.read',

        'SMTP.Send',
        'IMAP.AccessAsUser.All',
    ];

    /**
     * @return $this|Microsoft
     * @throws OAuthException
     */
    public function refresh()
    {
        $jwt = $this->generateClientSecret();

        $response = Http::asForm()->acceptJson()->withHeaders([
            'Authorization' => 'Bearer '.$this->token,
        ])->post('https://login.microsoftonline.com/common/oauth2/v2.0/token', [
            'grant_type' => 'refresh_token',
            'client_id' => config('services.microsoft.client_id'),
            // 'client_secret' => config('services.microsoft.client_secret'),
            'client_assertion_type' => 'urn:ietf:params:oauth:client-assertion-type:jwt-bearer',
            'client_assertion' => $jwt,
            'refresh_token' => $this->refreshToken,
            'scope' => implode(' ', $this->getScopes()),
        ]);

        if ($response->status() > 299) {
            throw OAuthException::invalidResponse('microsoft', $response);
        } else {
            // TODO: Handle error response
        }

        $data = $response->json();

        if (! $data['access_token'] ?? null) {
            throw OAuthException::invalidData('microsoft');
        }

        return new static($data['access_token'], $data['refresh_token'], Carbon::now()->addSeconds($data['expires_in']));
    }

    protected function getScopes()
    {
        $scopes = explode(' ', trim(config('services.microsoft.scopes')));

        return empty($scopes)
            ? $this->scopes
            : $scopes;
    }

    /**
     * @link https://docs.microsoft.com/en-us/azure/active-directory/develop/active-directory-certificate-credentials
     * @return string
     */
    protected function generateClientSecret()
    {
        return JWT::encode([
                'aud' => 'https://login.microsoftonline.com/common/oauth2/v2.0/token',
                'exp' => Carbon::now()->addMinutes(9.5)->timestamp,
                'nbf' => Carbon::now()->timestamp - 1,
                'iss' => config('services.microsoft.client_id'),
                'sub' => config('services.microsoft.client_id'),
                'jti' => Str::uuid()->toString(),
            ],
            file_get_contents(storage_path('keys/microsoft.pem')),
            'RS256',
            "950f408d-ff13-4975-8ed5-bbe5166525fc",
            [
                "alg" => "RS256",
                "typ" => "JWT",
                // @link https://stackoverflow.com/questions/50657463/how-to-obtain-value-of-x5t-using-certificate-credentials-for-application-authe/52625165
                // echo $(openssl x509 -in helpspot.pem -fingerprint -noout) | sed 's/SHA1 Fingerprint=//g' | sed 's/://g' | xxd -r -ps | base64
                "x5t" => "OcpSRGshA1mI8lcceLd0hrMD3vw=",
            ]
        );
    }
}
