<?php

namespace HS\Console\Commands;

use HS\Domain\Workspace\Document;

use Aws\S3\S3Client;
use Aws\S3\Exception\S3Exception;

use Illuminate\Console\Command;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class AttachmentsToS3Command extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'attachments:tos3 {bucket : S3 Bucket Name} {--path= : Attachment Path}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Save file-based attachments to S3 based on age.';

    /**
     * @var S3Client
     */
    protected $s3Client;

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        $path = ($this->option('path') && is_dir($this->option('path')))
            ? $this->option('path')
            : hs_setting('cHD_ATTACHMENT_LOCATION_PATH', storage_path('documents'));


        $fileDocs = $this->getFileDocuments();

        $notFound = [];
        $count = 0;
        foreach ($fileDocs as $file) {
            // If the file does not exist locally, it might be in S3 already (due to re-attachment logic).
            // If it's in S3 already, we'll update the document row to reflect that.
            // Otherwise we'll consider it "not found".
            if (! file_exists($path.$file->sFileLocation))
            {
                if ($this->existsInS3($file->sFileLocation)) {
                    Document::where('xDocumentId', $file->xDocumentId)->update([
                            'sFileLocation' => 's3://'.$this->argument('bucket').$file->sFileLocation, ]
                    );

                    $count++; // Only count if processed
                } else {
                    $notFound[] = $file;
                }
            } else {
                // If the file exists on disk drive, we'll process it normally (upload + unlink)
                $result = $this->upload($file->sFileLocation, $path.$file->sFileLocation);

                if ($result['ETag'] ?? null) {
                    Document::where('xDocumentId', $file->xDocumentId)->update([
                            'sFileLocation' => 's3://'.$this->argument('bucket').$file->sFileLocation, ]
                    );

                    // Delete the file when done uploading
                    @unlink($path.$file->sFileLocation);

                    $count++; // Only count if processed
                }
            }

            usleep(500000); // sleep .5 seconds to reduce hitting API rate limit
        }

        foreach ($notFound as $missing) {
            $this->error(
                sprintf("Can't find file '%s' on local file system", $path.$missing->sFileLocation)
            );
        }

        if (config('app.debug') && function_exists('xdebug_peak_memory_usage')) {
            $this->info("\nMemory Usage: ".round(xdebug_peak_memory_usage() / 1048576, 2).'MB');
        }
        $this->info($count . ' attachments out of ' . count($fileDocs) . ' found documents were successfully moved from the file system to S3.');
    }

    /**
     * Get documents with files saved to the disk.
     * @return Collection
     */
    protected function getFileDocuments()
    {
        return DB::table('HS_Documents')
            ->select(['xDocumentId', 'sFilename', 'sFileLocation'])
            ->whereNotNull('sFileLocation')
            ->where('sFileLocation', 'NOT LIKE', 's3://%')
            ->get();
    }

    /**
     * @link https://docs.aws.amazon.com/aws-sdk-php/v3/api/api-s3-2006-03-01.html#putobject
     * @param $key
     * @param $sourceFile
     * @return \Aws\Result
     */
    protected function upload($key, $sourceFile)
    {
        return $this->getS3()->putObject([
            'Bucket' => $this->argument('bucket'),
            'Key' => ltrim($key, '/'), // strip leading slash
            'SourceFile' => $sourceFile,
        ]);
    }

    /**
     * @link https://docs.aws.amazon.com/aws-sdk-php/v3/api/api-s3-2006-03-01.html#headobject
     * @param $key
     * @return bool
     */
    protected function existsInS3($key)
    {
        try {
            $result = $this->getS3()->headObject([
                'Bucket' => $this->argument('bucket'),
                'Key' => ltrim($key, '/'), // strip leading slash
            ]);

            return ($result['ContentLength'] ?? 0) > 0;
        } catch(\Exception $e)
        {
            if($e instanceof S3Exception) {
                return false;
            }

            Log::error($e);
        }

        return false;
    }

    /**
     * @return S3Client
     */
    public function getS3()
    {
        return $this->s3Client
            ?: new S3Client([
            'version' => 'latest',
            'region'  => 'us-east-1',
        ]);
    }
}
