<?php

namespace HS\Console\Commands;

use HS\IncomingMail\Loggers\DummyLogger;
use HS\IncomingMail\Message;
use HS\IncomingMail\Parsers\MimeDecode;
use Illuminate\Console\Command;
use HS\IncomingMail\Loggers\EchoLogger;
use HS\IncomingMail\Processors\ParserTags;
use HS\IncomingMail\Parsers\MailMimeParser;
use HS\IncomingMail\Processors\Attachments;

class ParseEmlCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'mail:parse {--eml=} {--dir=} {--oldparser} {--nogz}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Parse eml files for debugging. Pass the --eml= flag with the path to the file. --nogz if it is not encoded.';

    protected $nogz = false;

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $eml = $this->option('eml');
        $dir = $this->option('dir');
        $old = $this->option('oldparser');
        $this->nogz = $this->option('nogz');

        if (! $eml && ! $dir) {
            $this->error('Specify one of --eml (a single eml file) or --dir (a directory of eml files)');
            return 1;
        }

        $this->loadRequirements();

        if ($eml) {
            if (! file_exists(base_path($eml))) {
                $this->error('The file '.base_path($eml).' is not found.');
                return 1;
            }

            return $this->singleEml($eml, $old);
        }

        if ($dir) {
            return $this->directory($dir, $old);
        }
    }

    protected function directory($dir, $old)
    {
        if (!file_exists($dir) || ! is_dir($dir)) {
            $this->error('The location '.$dir.' does not exist or is not a directory.');
            return 1;
        }

        $directory = new \DirectoryIterator($dir);
        $errors = [];
        $this->comment("file_name,subject,from_name,from_email,parsed_message_length,text_message_length");
        foreach ($directory as $fileinfo) {
            /** @var \DirectoryIterator $fileinfo */
            if (! $fileinfo->isDot() && ($fileinfo->getExtension() == 'eml' || $fileinfo->getExtension() == 'gz')) {
                $message = new Message();
                if ($old) {
                    $decoder = new MimeDecode($message);
                } else {
                    $decoder = new MailMimeParser($message);
                }

                $body = $this->load_raw($fileinfo->getPathname());
                try {
                    $msg = $decoder->parse($body);
                    $this->info($this->csvFriendly($fileinfo->getFilename()).','.$this->csvFriendly($msg->getSubject()).','.$this->csvFriendly($msg->getFromName()).','.$this->csvFriendly($msg->getFromEmail()).','.strlen($msg->getBody()).','.strlen($msg->bodyText));
                } catch(\Exception $e) {
                    $errors[$fileinfo->getFilename()] = $e;
                }
            }
        }

        foreach($errors as $file => $error) {
            $this->error("[$file]: ".$error->getMessage());
        }

        return 0;
    }

    protected function csvFriendly($text)
    {
        return str_replace(',', ' ', $text);
    }

    protected function singleEml($eml, $old)
    {
        $body = $this->load_raw($eml);

        $message = new Message();
        if ($old) {
            $decoder = new MimeDecode($message);
        } else {
            $decoder = new MailMimeParser($message);
        }

        $msg = $decoder->parse($body);
        $body = $msg->getBody();

        $results = [
            ['subject', $msg->getSubject('no subject')],
            ['from name', $msg->getFromName()],
            ['from email', $msg->getFromEmail()],
            ['isForward', $msg->isForward() ? 'true' : 'false'],
            ['isHTML', $msg->isHtml() ? 'true' : 'false'],
            ['isImportant', $msg->isImportant() ? 'true' : 'false'],
        ];
        $this->table(['Title', 'Value'], $results);

        // custom tags
        $tags = new ParserTags($message);

        $results = [
            ['##hs_request_id', (string) $tags->hs_request_id($body, cHD_EMAILPREFIX)],
            ['##hs_customer_id', (string) $tags->hs_customer_id()],
            ['##hs_customer_firstname', (string) $tags->hs_customer_firstname()],
            ['##hs_customer_lastname', (string) $tags->hs_customer_lastname()],
            ['##hs_customer_phone', (string) $tags->hs_customer_phone()],
            ['##hs_customer_email', (string) $tags->hs_customer_email()],
            ['##hs_assigned_to', (string) $tags->hs_assigned_to()],
        ];

        $this->comment('+---------------------+---------------------------------------+');
        $this->comment('Parser Tags');
        $this->comment('+---------------------+---------------------------------------+');
        $this->table(['Field', 'Value'], $results);
        $this->comment('+---------------------+---------------------------------------+');
        $this->comment('Body after Parsing');
        $this->comment('+---------------------+---------------------------------------+');
        $this->line($msg->getBody());
        $this->comment('+---------------------+---------------------------------------+');
        $this->comment('Original Text Body');
        $this->comment('+---------------------+---------------------------------------+');
        $this->line($msg->bodyText);
        $this->comment('+---------------------+---------------------------------------+');
        $this->comment('Attachments');
        $this->comment('+---------------------+---------------------------------------+');
        $logger = new DummyLogger($eml);
        $attachments = new Attachments($msg, $logger);
        $msgFiles = $attachments->process();
        if ($msgFiles) {
            $this->line('Name / mimetype / content-id');
            foreach ($msgFiles as $file) {
                $this->line($file['name'].' / '.$file['mimetype'].' / '.$file['content-id']);
            }
        }
        $this->comment('+---------------------+---------------------------------------+');

        return 0;
    }

    protected function load_raw($file)
    {
        if ($this->nogz) {
            return file_get_contents(base_path($file));
        }
        return gzdecode(file_get_contents(base_path($file)));
    }

    protected function loadRequirements()
    {
        set_include_path(cBASEPATH.'/helpspot/pear');
        error_reporting(E_ALL ^ E_DEPRECATED);
        require_once base_path('/helpspot/helpspot/lib/api.lib.php');
        require_once base_path('/helpspot/helpspot/lib/util.lib.php');
        require_once base_path('/helpspot/helpspot/lib/utf8.lib.php');
        require_once base_path('/helpspot/helpspot/pear/Mail/helpspot_mimeDecode.php');
        if (! defined('cHD_EMAIL_REPLYABOVE')) {
            define('cHD_EMAIL_REPLYABOVE', '## Reply ABOVE THIS LINE to add a note to this request ##');
        }
        if (! defined('lg_outlookseparator')) {
            define('lg_outlookseparator', '/-----\s*Original Message\s*-----/');
        }
        if (! defined('cHD_STRIPHTML')) {
            define('cHD_STRIPHTML', '2');
        }
        if (! defined('cHD_CUSTOMER_ID')) {
            define('cHD_CUSTOMER_ID', '1');
        }
        if (! defined('cHD_EMAILPREFIX')) {
            define('cHD_EMAILPREFIX', '');
        }
    }
}
