<?php

namespace HS\IncomingMail\Mailboxes;

use Google\Service\Gmail\ModifyMessageRequest;
use Google_Client;
use Google_Service_Gmail;
use HS\Mailbox as MailboxModel;
use Google\Service\Gmail\ListMessagesResponse;

class Gmail implements Mailbox
{

    /**
     * @var MailboxModel
     */
    private $mailbox;

    /**
     * @var Google_Service_Gmail
     */
    private $service;

    /**
     * @var ListMessagesResponse
     */
    private $results;

    /**
     * @throws \HS\Auth\OAuth\OAuthException
     */
    public function __construct(MailboxModel $mailbox)
    {
        $this->mailbox = $mailbox;

        $client = $this->getClient();
        $tokens = $this->mailbox->getTokens();

        $client->setAccessToken([
            'access_token' => $tokens->token(),
            'refresh_token' => $tokens->refreshToken(),
            'expires_in' => (string) $tokens->expiresAt(),
        ]);

        $this->service = new Google_Service_Gmail($client);
    }

    /**
     * Get the Google client
     * @return Google_Client
     */
    protected function getClient()
    {
        return new Google_Client([
            'clientId'      => config('services.google.client_id'),
            'client_id'     => config('services.google.client_id'),
            'clientSecret'  => config('services.google.client_secret'),
            'client_secret' => config('services.google.client_secret'),
//            'scopes'        => explode(' ', trim(config('services.google.scopes'))),
            'accessType'    => 'offline',
        ]);
    }

    /**
     * Connect to mailbox and get some messages.
     * @return mixed|void
     */
    public function connect()
    {
        $this->results = $this->service->users_messages->listUsersMessages('me', ['includeSpamTrash' => false, 'labelIds' => 'INBOX']);
    }

    /**
     * @inheritDoc
     */
    public function messageCount()
    {
        return $this->results->getResultSizeEstimate();
    }

    /**
     * @inheritDoc
     */
    public function getMessage($id)
    {
        $message = $this->service->users_messages->get('me', $id, ['format' => 'raw']);
        return base64_decode(strtr($message->getRaw(), '-_', '+/'));
    }

    /**
     * @inheritDoc
     */
    public function delete($id)
    {
        return $this->archive($id);
    }

    /**
     * @param $id
     * @return \Google\Service\Gmail\Message|mixed
     */
    public function archive($id)
    {
        $request = new ModifyMessageRequest();
        $request->setRemoveLabelIds(['UNREAD', 'INBOX']);
        return $this->service->users_messages->modify('me', $id, $request);
    }

    /**
     * @inheritDoc
     */
    public function expunge()
    {
        return true;
    }

    /**
     * Get the uid based on the message number
     *
     * @param $message
     * @return int
     */
    public function getUid($message)
    {
        return $message->getId();
    }

    /**
     * @inheritDoc
     */
    public function getMessages()
    {
        $pageToken = null;
        $messages = [];
        $opt_param = ['includeSpamTrash' => false, 'labelIds' => 'INBOX'];
        do {
            try {
                if ($pageToken) {
                    $opt_param['pageToken'] = $pageToken;
                }
                $messagesResponse = $this->service->users_messages->listUsersMessages('me', $opt_param);
                if ($messagesResponse->getMessages()) {
                    $messages = array_merge($messages, $messagesResponse->getMessages());
                    $pageToken = $messagesResponse->getNextPageToken();
                }
            } catch (\Exception $e) {
                print 'An error occurred: ' . $e->getMessage();
            }
        } while ($pageToken);

        return $messages;
    }
}
