<?php

namespace HS\IncomingMail\Mailboxes;

use Microsoft\Graph\Graph;
use Microsoft\Graph\Model;
use HS\Mailbox as MailboxModel;
use HS\Auth\OAuth\OAuthException;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;
use Microsoft\Graph\Http\GraphResponse;
use GuzzleHttp\Exception\GuzzleException;

class Microsoft implements Mailbox
{

    /**
     * @var Graph
     */
    private $service;

    /**
     * @var MailboxModel
     */
    private $mailbox;

    /**
     * @var GraphResponse|mixed
     */
    private $user;

    /**
     * @var GraphResponse|mixed
     */
    private $results;

    /**
     * @var string
     */
    private $_token;

    /**
     * @param MailboxModel $mailbox
     * @throws GuzzleException
     * @throws OAuthException
     */
    public function __construct(MailboxModel $mailbox)
    {
        $this->mailbox = $mailbox;
        $this->_token = $this->mailbox->getTokens()->token();
        $graph = new Graph();
        $graph->setAccessToken($this->_token);
        $this->service = $graph;

        try {
            $this->user = $graph->createRequest('GET', '/me')
                ->setReturnType(Model\User::class)
                ->execute();
        } catch (\Exception $e) {
            Log::error(new \Exception('Microsoft Connection Failure'), [
                'xMailbox' => $this->mailbox->xMailbox,
            ]);
        }
    }

    /**
     * @inheritDoc
     */
    public function connect()
    {
        $messageQueryParams = [
            // Only return Subject, ReceivedDateTime, and From fields
            "\$select" => "id",
            // Sort by ReceivedDateTime, newest first
            "\$orderby" => "receivedDateTime DESC",
            // Return at most cHD_EMAILS_MAX_TO_IMPORT results
            "\$top" => hs_setting('cHD_EMAILS_MAX_TO_IMPORT', '10')
        ];
        $getMessagesUrl = '/me/mailfolders/inbox/messages?'.http_build_query($messageQueryParams);
        $this->results = $this->service->createRequest('GET', $getMessagesUrl)
            ->addHeaders(array ('X-AnchorMailbox' => $this->user->getMail()))
            ->setReturnType(Model\Message::class)
            ->execute();
    }

    /**
     * @inheritDoc
     */
    public function messageCount()
    {
        return count($this->results);
    }

    /**
     * @inheritDoc
     */
    public function getMessage($id)
    {
        $getMessagesUrl = 'https://graph.microsoft.com/v1.0/me/messages/'.$id.'/$value';
        $response = Http::withToken($this->_token)->get($getMessagesUrl);
        return $response->body();
    }

    /**
     * @inheritDoc
     */
    public function getUid($message)
    {
        return $message->getId();
    }

    /**
     * Delete a message
     * https://docs.microsoft.com/en-us/graph/api/message-move?view=graph-rest-1.0&tabs=http
     * @param string $id
     * @return bool
     */
    public function delete($id)
    {
        $getMessagesUrl = 'https://graph.microsoft.com/v1.0/me/messages/'.$id.'/move';
        $response = Http::withToken($this->_token)->post($getMessagesUrl, ['destinationId' => 'deleteditems']);
        return $response->ok();
    }

    /**
     * Archive a message
     * https://docs.microsoft.com/en-us/graph/api/message-move?view=graph-rest-1.0&tabs=http
     * https://docs.microsoft.com/en-us/graph/api/resources/mailfolder?view=graph-rest-1.0
     *
     * @param string $id
     * @return bool
     */
    public function archive($id)
    {
        $getMessagesUrl = 'https://graph.microsoft.com/v1.0/me/messages/'.$id.'/move';
        $response = Http::withToken($this->_token)->post($getMessagesUrl, ['destinationId' => 'archive']);
        return $response->ok();
    }

    /**
     * @inheritDoc
     */
    public function expunge()
    {
        return true;
    }

    /**
     * @inheritDoc
     */
    public function getMessages()
    {
        return $this->results;
    }
}
