<?php

namespace HS\Mail;

use HS\Mailbox;
use HS\Mail\Mailer\SMTP;
use HS\Mail\Mailer\SmtpSettings;
use HS\Mail\Transports\Google;
use HS\Mail\Transports\Microsoft;

use Swift_Mailer;
use Swift_SmtpTransport;
use Swift_SendmailTransport;
use Swift_Plugins_LoggerPlugin;
use Swift_Plugins_Loggers_ArrayLogger;

class HelpspotMailer
{
    /**
     * @var Swift_Plugins_Loggers_ArrayLogger
     * usage: `HelpspotMailer::$logger->dump()`
     */
    public static $logger;

    /**
     * @var int
     */
    private $mailboxId;

    /**
     * HelpspotMailer constructor.
     * @param $mailboxId
     */
    public function __construct($mailboxId=0)
    {
        $this->mailboxId = $mailboxId;
    }

    /**
     * @param $mailboxId
     * @return Mailer
     * @throws \Exception
     */
    public static function via($mailboxId)
    {
        $mailer = new static($mailboxId);

        $swift = $mailer->generateSwift();

        $swift->registerPlugin(new Swift_Plugins_LoggerPlugin($mailer->getLogger()));

        return $mailer->generateMailer(
            $swift,
            $mailer->getMailerName($swift->getTransport())
        );
    }

    /**
     * Get SMTP mailer, used for testing SMTP from
     * Admin > Settings > Email Integration
     * @param Mailbox|null $mailbox
     * @return Mailer
     * @throws \Exception
     */
    public static function smtpTest(Mailbox $mailbox)
    {
        $mailer = new static;
        $swift = $mailer->smtpTransport(SMTP::fromHelpSpotSettings($mailbox));
        $swift->registerPlugin(new Swift_Plugins_LoggerPlugin($mailer->getLogger()));
        return $mailer->generateMailer($swift);
    }

    /**
     * Get sendmail mailer, used for testing PHP Mail from
     * Admin > Settings > Email Integration
     * @return Mailer
     * @throws \Exception
     */
    public static function sendmailTest()
    {
        $mailer = new static;
        $swift = $mailer->mailTransport();
        $swift->registerPlugin(new Swift_Plugins_LoggerPlugin($mailer->getLogger()));
        return $mailer->generateMailer($swift, 'sendmail');
    }

    /**
     * @param $swift
     * @param string $name
     * @return Mailer
     */
    public function generateMailer($swift, $name='smtp')
    {
        return with(new Mailer($name, app('view'), $swift, app('events')), function(Mailer $mailer) {
            $mailer->setQueue(app('queue'));
            return $mailer;
        });
    }

    /**
     * @return Swift_Mailer
     * @throws \Exception
     */
    public function generateSwift()
    {
        /** @var Mailbox $mailbox */
        $mailbox = Mailbox::find($this->mailboxId);

        // If a mailbox was provided, and the mailbox has custom SMTP
        // then use the custom SMTP from the mailbox
        if ($mailbox && $mailbox->hasCustomSmtp()) {
            return $this->smtpTransport($mailbox->smtp());
        }

        // If a mailbox was provided, and is an oAuth mailbox
        // use the oAuth credentials for that mailbox
        // Note that in this case, $this->mailboxId is set to the mailbox ID already
        if ($mailbox && $mailbox->isOAuth()) {
            $transport = sprintf('%sTransport', $mailbox->sMailboxType);
            return $this->$transport();
        }

        // Get default configured transport
        // If the default is an oAuth mailbox (not smtp, nor sendmail),
        // then we have to set $this->mailboxId to the correct mailbox
        if(config()->get('mail.default') != 'smtp' && config()->get('mail.default') != 'sendmail') {
            // e.g if mail.microsoft.mailbox isset and has a value...
            if (config()->get('mail.mailers.'.config()->get('mail.default').'.mailbox'))
            {
                $this->mailboxId = config()->get('mail.mailers.'.config()->get('mail.default').'.mailbox');
            } else {
                throw new \Exception('oAuth mailbox in config path `mail.mailers.'.config()->get('mail.default').'.mailbox` has no configured mailbox ID');
            }
        }

        $transport = sprintf('%sTransport', config()->get('mail.default'));
        return $this->$transport();
    }

    /**
     * @return Swift_Mailer
     */
    protected function mailTransport()
    {
        return new Swift_Mailer(new Swift_SendmailTransport);
    }

    /**
     * @param SMTP|null $smtp
     * @return Swift_Mailer
     * @throws \Exception
     */
    protected function smtpTransport(SMTP $smtp=null)
    {
        $smtp = ($smtp)
            ?: (new SmtpSettings($this->mailboxId))->settings();

        return with(new Swift_SmtpTransport($smtp->host, $smtp->port, $smtp->encryption), function($transport) use($smtp) {
            if( $smtp->auth ) {
                $transport->setUsername($smtp->username);
                $transport->setPassword($smtp->password);
            }
            if ($smtp->helo) {
                $transport->setLocalDomain($smtp->helo);
            }
            return new Swift_Mailer($transport);
        });
    }

    /**
     * @return Swift_Mailer
     */
    protected function microsoftTransport()
    {
        return new Swift_Mailer(new Microsoft($this->mailboxId));
    }

    /**
     * @return Swift_Mailer
     */
    protected function googleTransport()
    {
        return new Swift_Mailer(new Google($this->mailboxId));
    }

    /**
     * @return Swift_Plugins_Loggers_ArrayLogger
     */
    protected function getLogger()
    {
        $logger = new Swift_Plugins_Loggers_ArrayLogger();
        self::$logger = $logger;

        return $logger;
    }

    protected function getMailerName($transport)
    {
        if ($transport instanceof Swift_SmtpTransport) {
            return 'smtp';
        }

        if ($transport instanceof Microsoft)
        {
            return 'microsoft';
        }

        if ($transport instanceof Google)
        {
            return 'google';
        }

        if ($transport instanceof Swift_SendmailTransport)
        {
            return 'sendmail';
        }

        // Likely sendmail
        return config()->get('mail.default');
    }
}
