<?php

namespace HS\Mail\Transports;

use HS\Mailbox;
use Swift_Mime_SimpleMessage;

use Google\Client;
use Google\Service\Gmail;
use Google\Http\MediaFileUpload;
use Google\Service\Gmail\Message;

use Illuminate\Mail\Transport\Transport;

class Google extends Transport
{
    /**
     * 5 MB in bytes
     */
    const FIVE_MEGABYTES = 5000000;

    /**
     * @var Mailbox
     */
    protected $mailbox;

    /**
     * Microsoft constructor.
     * @param int $xMailbox
     */
    public function __construct($xMailbox)
    {
        $this->mailbox = Mailbox::find($xMailbox);
    }

    /**
     * @link handle-attachments https://stackoverflow.com/a/46733040/1412984
     * @param Swift_Mime_SimpleMessage $message
     * @param null $failedRecipients
     * @return int|void
     */
    public function send(Swift_Mime_SimpleMessage $message, &$failedRecipients = null)
    {
        $client = new Client([
            'client_id'     => config('services.google.client_id'),
            'client_secret' => config('services.google.client_secret'),
        ]);
        $client->setAccessToken($this->googleAccessToken());
        $gmail = new Gmail($client);

        $encodedMIME = base64_encode($message->toString());

        $msg = new Message;
        $messageSize = strlen($encodedMIME);

        // We check the size against the base64-encoded version here on purpose,
        // not the raw message ($message->toString()). It's possible the base64-encoded
        // version is over 5mb but the mime string is under. That shoulid be OK.
        if($messageSize >= self::FIVE_MEGABYTES) {
            $client->setDefer(true);
            $request = $gmail->users_messages->send('me', $msg, ['uploadType' => 'resumable']);

            $media = new MediaFileUpload(
                $client,
                $request,
                'message/rfc822',
                $message->toString(), // We don't pass it the base64 encoded mime
                true,
                1 * 1000 * 1000 // 1MB
            );
            $media->setFileSize(strlen($message->toString()));

            $status = false;
            while (! $status) {
                $status = $media->nextChunk();
            }

            // $client->setDefer(false);
            // $googleMessageId = $status->getId();
        } else {
            $msg->setRaw($encodedMIME);
            $gmail->users_messages->send('me', $msg);
        }
    }

    protected function googleAccessToken()
    {
        $tokens = $this->mailbox->getTokens();

        return [
            'access_token' => $tokens->token(),
            'refresh_token' => $tokens->refreshToken(),
            'expires_in' => $tokens->expiresAt()->timestamp,
        ];
    }
}
