<?php

namespace HS\Mail\Transports;

use HS\Mailbox;

use Swift_Mime_SimpleMessage;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;
use Illuminate\Mail\Transport\Transport;

class Microsoft extends Transport
{
    /**
     * 3 MB in bytes
     */
    const THREE_MEGABYTES = 3000000;

    /**
     * @var Mailbox
     */
    protected $mailbox;

    /**
     * @var string
     */
    protected $token;

    /**
     * Microsoft constructor.
     * @param int $xMailbox
     */
    public function __construct($xMailbox)
    {
        $this->mailbox = Mailbox::find($xMailbox);
    }

    /**
     * @link send-mail https://docs.microsoft.com/en-us/graph/api/user-sendmail?view=graph-rest-1.0&tabs=http
     * @link create-message https://docs.microsoft.com/en-us/graph/api/user-post-messages?view=graph-rest-1.0&tabs=http#draft-message
     * @link message-resource https://docs.microsoft.com/en-us/graph/api/resources/message?view=graph-rest-1.0
     * @link add-attachment https://docs.microsoft.com/en-us/graph/api/message-post-attachments?view=graph-rest-1.0&tabs=http#add-small-attachments
     * @link add-large-attachment https://docs.microsoft.com/en-us/graph/outlook-large-attachments?tabs=http#add-large-attachments
     * @link message-send https://docs.microsoft.com/en-us/graph/api/message-send?view=graph-rest-1.0&tabs=http#send-message
     * @param Swift_Mime_SimpleMessage $message
     * @param null $failedRecipients
     * @return int|void
     * @throws \HS\Auth\OAuth\OAuthException|\Illuminate\Http\Client\RequestException
     */
    public function send(Swift_Mime_SimpleMessage $message, &$failedRecipients = null)
    {
        $this->token = $this->microsoftAccessToken();

        // 1. Create a message draft
        $response = Http::withToken($this->token)
            ->asJson()
            ->post('https://graph.microsoft.com/v1.0/me/messages', [
                'subject' => $message->getSubject(),
                'body' => $this->itemBody($message),
                // 'from' => $this->emailAddresses($message->getFrom()), // TODO: This may need filling out if HelpSpot sets the From as a different email
                'bccRecipients' => $this->emailAddresses($message->getBcc()),
                'ccRecipients' => $this->emailAddresses($message->getCc()),
                'toRecipients' => $this->emailAddresses($message->getTo()),
                'hasAttachments' => $this->hasAttachments($message), // Only if there are non-inline attachments
                'internetMessageHeaders' => $this->headers($message),
                'internetMessageId' => optional($message->getHeaders()->get('message-id'))->getFieldBody(),
                'conversationId' => optional($message->getHeaders()->get('references'))->getFieldBody(),
            ])->throw();

        $draftedMessage = $response->json();

        // 2. Upload attachments (handling attachments over 3MB specially)
        // todo:  Needs to correctly handle inline images (CID)
        $this->addAttachmentsToMessage($message, $draftedMessage);

        // 3. Send drafted message
        $response = Http::withToken($this->token)
            ->post('https://graph.microsoft.com/v1.0/me/messages/'.$draftedMessage['id'].'/send')
            ->throw();

        if ($response->status() != 202) {
            Log::info('Microsoft email failed to send', [
                'status' => $response->status(),
                'conversationId' => $message->getHeaders()->get('references')->getFieldBody(),
            ]);
        }
    }

    /**
     * @return string
     */
    protected function microsoftAccessToken()
    {
        return $this->mailbox->getTokens()->token();
    }

    /**
     * Return an itemBody resource type
     * @link https://docs.microsoft.com/en-us/graph/api/resources/itembody?view=graph-rest-1.0
     * @param $message
     * @return array
     */
    protected function itemBody($message)
    {
        return [
            'contentType' => str_replace('text/', '', $message->getBodyContentType()),
            'content' => (string)$message->getBody(),
        ];
    }

    /**
     * Return a collection of emailAddress resource types
     * @link https://docs.microsoft.com/en-us/graph/api/resources/emailaddress?view=graph-rest-1.0
     * @param $addresses
     * @return array
     */
    protected function emailAddresses($addresses)
    {
        $formattedAddresses = [];

        foreach($addresses ?? [] as $email => $name)
        {
            $formattedAddresses[] = [
                'emailAddress' => [
                    'address' => $email,
                    'name' => $name,
                ],
            ];
        }

        return $formattedAddresses;
    }

    protected function headers(Swift_Mime_SimpleMessage $message)
    {
        $messageHeaders = $message->getHeaders();

        // All headers here must start with "x-"
        $headersToGet = [
            'X-Helpspot',
            'X-Mailer',
            // 'In-Reply-To', // See @link https://stackoverflow.com/questions/55945615/cannot-pass-in-reply-to-parameter-to-microsoft-graph-sendmail
            // 'References',
            'X-Helpspot-Request-Id',
        ];

        $headers = [];
        foreach($headersToGet as $header) {
            if ($messageHeaders->has($header))
            {
                $headers[] = [
                    'name' => $header,
                    'value' => $messageHeaders->get($header)->getFieldBody(),
                ];
            }
        }

        return $headers;
    }

    protected function hasAttachments(Swift_Mime_SimpleMessage $message)
    {
        foreach($message->getChildren() as $child)
        {
            if ($child instanceof \Swift_Attachment) {
                return true;
            }
        }

        return false;
    }

    protected function addAttachmentsToMessage(Swift_Mime_SimpleMessage $message, $draftedMessage)
    {
        foreach($message->getChildren() as $child)
        {
            // File attachment || embedded image
            if ($child instanceof \Swift_Attachment || $child instanceof \Swift_Image)
            {
                if (! $size = $child->getSize()) {
                    // Should only hit this is UploadAttachment is used
                    // However that's likely unused
                    $size = strlen($child->getBody());
                }

                // File attachment
                if ($size >= self::THREE_MEGABYTES) {
                    // uploadSession
                    $this->uploadLargeAttachment($child, $draftedMessage, $size);
                } else {
                    // regular attachment
                    $this->uploadAttachment($child, $draftedMessage, $size);
                }
            }
        }
    }

    /**
     * @link attachment-resource https://docs.microsoft.com/en-us/graph/api/resources/attachment?view=graph-rest-1.0
     * @link file-attachment-resource https://docs.microsoft.com/en-us/graph/api/resources/fileattachment?view=graph-rest-1.0
     * @param \Swift_Attachment|\Swift_Image $child
     * @param array $draftedMessage
     * @param int $size
     * @return \GuzzleHttp\Promise\PromiseInterface|\Illuminate\Http\Client\Response
     * @throws \Illuminate\Http\Client\RequestException
     */
    protected function uploadAttachment($child, $draftedMessage, $size)
    {
        return Http::withToken($this->token)
            ->asJson()
            ->post('https://graph.microsoft.com/v1.0/me/messages/'.$draftedMessage['id'].'/attachments', [
                '@odata.type' => '#microsoft.graph.fileAttachment',
                'name' => $child->getFilename(),
                'contentType' => $child->getContentType(),
                'isInline' => $child instanceof \Swift_Image,
                'size' => $size,
                'contentBytes' => base64_encode($child->getBody()), // Docs suggest they don't want base64 encoded bytes...
                'contentId' => $child->getId(),
            ])->throw();
    }

    /**
     * @param \Swift_Attachment|\Swift_Image $child
     * @param array $draftedMessage
     * @param bytes $size
     * @throws \Illuminate\Http\Client\RequestException
     */
    protected function uploadLargeAttachment($child, $draftedMessage, $size)
    {
        $response = Http::withToken($this->token)
            ->asJson()
            ->post('https://graph.microsoft.com/v1.0/me/messages/'.$draftedMessage['id'].'/attachments/createUploadSession', [
                'AttachmentItem' => [
                    'attachmentType' => 'file',
                    'name' => $child->getFilename(),
                    'contentType' => $child->getContentType(),
                    'isInline' => $child instanceof \Swift_Image,
                    'size' => $size,
                    'contentId' => $child->getId(),
                ],
            ])->throw();

        $session = $response->json();

        // Sigh
        $body = $child->getBody();
        $chunks = str_split($body, self::THREE_MEGABYTES);

        foreacH($chunks as $key => $chunk)
        {
            $start = $key * self::THREE_MEGABYTES;
            $end = $key * self::THREE_MEGABYTES + strlen($chunk) - 1;
            HTTP::withBody($chunk, 'application/octet-stream')
                ->withHeaders([
                    'Content-Length' => strlen($chunk),
                    'Content-Range' => "bytes $start-$end/$size"
                ])
                ->put($session['uploadUrl'])
                ->throw();
        }

    }
}
