<?php

namespace HS;


use HS\Mail\Mailer\SMTP;
use HS\Auth\OAuth\Google;
use HS\Auth\OAuth\Microsoft;
use HS\Auth\OAuth\Credentials;
use HS\IncomingMail\FetchMailConcurrency;

use Illuminate\Database\Eloquent\Model;

class Mailbox extends Model
{
    use FetchMailConcurrency;

    protected $table = 'HS_Mailboxes';

    protected $primaryKey = 'xMailbox';

    public $timestamps = false;

    protected $guarded = [];

    protected $appends = [];

    protected $casts = [
        'fArchive' => 'boolean',
    ];

    public static function getActive()
    {
        return self::active()->get();
    }

    public static function getActiveCustom()
    {
        return self::active()
            ->custom()
            ->get();
    }

    public function scopeActive($query)
    {
        return $query->where('fDeleted', 0);
    }

    public function scopeCustom($query)
    {
        return $query->where('sMailboxType', 'custom');
    }

    /**
     * A human-readable identifier for mailbox
     * @return string
     */
    public function identify()
    {
        return vsprintf("(%s) %s :: %s", [
            $this->getKey(),
            $this->sHostname,
            $this->Username,
        ]);
    }

    public function getSSMTPSettingsAttribute($value)
    {
        return (hs_empty($value) ? false : hs_unserialize($value));
    }

    /**
     * @return bool
     */
    public function hasCustomSmtp()
    {
        return (bool) $this->sSMTPSettings;
    }

    public function isOAuth()
    {
        return $this->sMailboxType != 'custom';
    }

    public function smtp()
    {
        return SMTP::fromHelpSpotSettings($this);
    }

    public function scopeForListing($query)
    {
        $concatField = (config('database.default') == 'mysql')
            ? "CONCAT_WS('@','HS_Mailboxes.sUsername', 'HS_Mailboxes.sHostname')"  // mysql
            : "CAST(HS_Mailboxes.sUsername AS NVARCHAR(max)) + '@' + CAST(HS_Mailboxes.sHostname AS NVARCHAR(max))"; // sqlsrv

        return $query->select(['HS_Mailbox.*', $concatField]);
    }

    public function scopeArchived($query, $archived=false)
    {
        $value = ($archived)
            ? '1'
            : '0';

        return $query->where('fDeleted', $value);
    }

    /**
     * Encode oAuth Credentials
     * @return string
     */
    public function encodedOAuthCredentials()
    {
        return ($this->sOAuthCredentials)
            ? base64_encode(json_encode($this->sOAuthCredentials))
            : '';
    }

    /**
     * @param string $encodedCredentials Base64-encoded JSON string of oauth credentials
     * @return array ['token' => 'xxx', 'refreshToken' => 'yyy', 'expiresIn' => '2019-02-01T03:45:27.612584Z'] (Date is result of `Carbon::toIsoString()`)
     */
    public static function decodeOAuthCredentials($encodedCredentials)
    {
        return json_decode(base64_decode($encodedCredentials), true);
    }

    public static function validSort($sortBy)
    {
        if (! in_array($sortBy, ['sReplyName', 'sHostname', 'sUsername'])) {
            return 'sReplyName';
        }

        return $sortBy;
    }

    public static function validOrder($order)
    {
        $order = strtolower($order);
        if (! in_array($order, ['asc', 'desc'])) {
            return 'asc';
        }

        return $order;
    }

    public function getSPasswordAttribute()
    {
        if ($this->attributes['sPassword']) {
            return decrypt($this->attributes['sPassword']);
        }

        return null;
    }

    public function setSPasswordAttribute($value)
    {
        $this->attributes['sPassword'] = encrypt($value);
    }

    public function getSOAuthCredentialsAttribute()
    {
        if ($this->attributes['sOAuthCredentials']) {
            return decrypt($this->attributes['sOAuthCredentials']);
        }

        return null;
    }

    public function setSOAuthCredentialsAttribute($value)
    {
        $this->attributes['sOAuthCredentials'] = encrypt($value);
    }

    public function getTokens()
    {
        if ($this->sMailboxType == 'google') {
            return $this->refreshAndStore(Google::fromEncoded($this->sOAuthCredentials));
        }

        if ($this->sMailboxType == 'microsoft') {
            return $this->refreshAndStore(Microsoft::fromEncoded($this->sOAuthCredentials));
        }

        return null;
    }

    /**
     * @param Credentials $credentials
     * @return Credentials
     */
    protected function refreshAndStore(Credentials $credentials)
    {
        $newTokens = $credentials->refresh();
        $this->sOAuthCredentials = $newTokens->toEncoded();
        return $newTokens;
    }

}
